<?php

namespace Drupal\commerce_order;

/**
 * Provides common logic for processing and transforming adjustments.
 */
interface AdjustmentTransformerInterface {

  /**
   * Combines, sorts, and rounds the given adjustments.
   *
   * @param \Drupal\commerce_order\Adjustment[] $adjustments
   *   The adjustments.
   *
   * @return \Drupal\commerce_order\Adjustment[]
   *   The processed adjustments.
   */
  public function processAdjustments(array $adjustments);

  /**
   * Combines adjustments with the same type and source ID.
   *
   * For example, all tax adjustments generated by the same tax type
   * will be combined into a single adjustment, which can then be shown
   * in total summaries.
   *
   * @param \Drupal\commerce_order\Adjustment[] $adjustments
   *   The adjustments.
   *
   * @return \Drupal\commerce_order\Adjustment[]
   *   The combined adjustments.
   */
  public function combineAdjustments(array $adjustments);

  /**
   * Sorts adjustments by their type's weight.
   *
   * For example, tax adjustments will be placed after promotion adjustments,
   * because the tax adjustment type has a higher weight than the promotion
   * one, as defined in commerce_order.commerce_adjustment_types.yml.
   *
   * @param \Drupal\commerce_order\Adjustment[] $adjustments
   *   The adjustments.
   *
   * @return \Drupal\commerce_order\Adjustment[]
   *   The sorted adjustments.
   */
  public function sortAdjustments(array $adjustments);

  /**
   * Rounds adjustments to their currency precision.
   *
   * For example, USD adjustments will be rounded to 2 decimals.
   *
   * @param \Drupal\commerce_order\Adjustment[] $adjustments
   *   The adjustments.
   * @param int $mode
   *   The rounding mode. One of the following constants: PHP_ROUND_HALF_UP,
   *   PHP_ROUND_HALF_DOWN, PHP_ROUND_HALF_EVEN, PHP_ROUND_HALF_ODD.
   *
   * @return \Drupal\commerce_order\Adjustment[]
   *   The rounded adjustments.
   */
  public function roundAdjustments(array $adjustments, $mode = PHP_ROUND_HALF_UP);

  /**
   * Rounds an adjustment to its currency precision.
   *
   * For example, a USD adjustment will be rounded to 2 decimals.
   *
   * @param \Drupal\commerce_order\Adjustment $adjustment
   *   The adjustment.
   * @param int $mode
   *   The rounding mode. One of the following constants: PHP_ROUND_HALF_UP,
   *   PHP_ROUND_HALF_DOWN, PHP_ROUND_HALF_EVEN, PHP_ROUND_HALF_ODD.
   *
   * @return \Drupal\commerce_order\Adjustment
   *   The rounded adjustment.
   */
  public function roundAdjustment(Adjustment $adjustment, $mode = PHP_ROUND_HALF_UP);

}
